## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 8,
  fig.height = 6,
  dpi = 150
)

## ----setup--------------------------------------------------------------------
library(rbranding)
library(ggplot2)

# Suggested packages for enhanced functionality
# library(plotly)  # For interactive plots
# library(sysfonts)  # For custom fonts
# library(showtext)  # For custom fonts

## -----------------------------------------------------------------------------
## Use a temporary directory as the knit root for the entire document.
## This avoids calling setwd()/on.exit() in the chunk and ensures
## subsequent chunks are evaluated with `temp_dir` as their working dir.
temp_dir <- tempdir()
knitr::opts_knit$set(root.dir = temp_dir)

# (Optional) store the original working directory for interactive use only
# Note: during knitting, chunks will be evaluated with root.dir set to temp_dir,
# so we explicitly copy/write files into that directory below.
old_wd <- getwd()

# Initialize branding (creates rbranding_config.yml and placeholder _brand.yml)
brand_init()

# Get the latest brand file from the repository
get_brand_public()

# to install these files directly to your working directory (the knit root):
get_template("ggplot2")

## -----------------------------------------------------------------------------
# In a real project, you would have a _brand.yml file in your working directory
# For this demo, we'll use the package's example brand file
brand_file <- system.file("brand_files", "_brand.yml", package = "rbranding")
if (brand_file != "") {
  # copy the example brand file and logos into the knit root (temp_dir)
  file.copy(brand_file, file.path(temp_dir, "_brand.yml"))
  # Copy logo files as well
  logo_files <- list.files(system.file("brand_files", package = "rbranding"), 
                          pattern = "*.png", full.names = TRUE)
  file.copy(logo_files, temp_dir)
  # Use a relative path for later chunks (they run with root.dir=temp_dir)
  brand_file <- "_brand.yml"
} else {
  # Fallback to a basic brand configuration for demonstration
  brand_config <- "
meta:
  name:
    full: Example Organization
    short: EO

color:
  palette:
    primary: '#1c8478'
    secondary: '#4e2d53'
    accent: '#474747'
  foreground: black
  background: white
  primary: primary
  secondary: secondary

typography:
  fonts:
    - family: Open Sans
      source: google
  base: Open Sans
"
  # write a fallback brand file into the knit root
  writeLines(brand_config, file.path(temp_dir, "_brand.yml"))
  brand_file <- "_brand.yml"
}

cat("Using brand file:", brand_file)

## -----------------------------------------------------------------------------
# Set the brand theme
brand_set_ggplot(brand_file)

## ----basic-plot, fig.alt="Branded scatterplot of car weight vs. miles per gallon"----
# Create a basic scatter plot
p1 <- ggplot(mtcars, aes(x = mpg, y = wt)) +
  geom_point(aes(color = factor(cyl)), size = 3) +
  labs(
    title = "Car Weight vs. Miles per Gallon",
    subtitle = "Data from the 1974 Motor Trend magazine",
    x = "Miles per Gallon",
    y = "Weight (1000 lbs)",
    color = "Cylinders"
  ) +
  theme(legend.position = "bottom")

print(p1)

## ----bar-plot, fig.alt="Branded bar plot of car count by cylinders and gears"----
# Create a bar plot
p2 <- ggplot(mtcars, aes(x = factor(cyl), fill = factor(gear))) +
  geom_bar(position = "dodge") +
  labs(
    title = "Car Count by Cylinders and Gears",
    x = "Number of Cylinders",
    y = "Count",
    fill = "Gears"
  ) +
  theme(legend.position = "bottom")

print(p2)

## ----logo-plot, eval=FALSE----------------------------------------------------
# # Add logo to the plot (requires logo in brand.yml and png package)
# p1_with_logo <- p1 + brand_add_logo(x = 0.9, y = 0.1, size = 0.05)
# print(p1_with_logo)

## ----plotly-example, eval=FALSE-----------------------------------------------
# library(plotly)
# 
# # Convert ggplot to interactive plotly chart
# p1_interactive <- ggplotly(p1, tooltip = c("x", "y", "colour"))
# p1_interactive

## ----advanced-theme, fig.alt="Branded scatterplot of engine performance analysis"----
# Customize theme elements while keeping brand colors
p3 <- ggplot(mtcars, aes(x = hp, y = mpg, size = wt)) +
  geom_point(alpha = 0.7) +
  scale_size_continuous(range = c(2, 8)) +
  labs(
    title = "Engine Performance Analysis",
    subtitle = "Relationship between horsepower, fuel efficiency, and weight",
    x = "Horsepower",
    y = "Miles per Gallon",
    size = "Weight (1000 lbs)"
  ) +
  theme(
    plot.title = element_text(size = 16, face = "bold"),
    plot.subtitle = element_text(size = 12, face = "italic"),
    legend.position = "right"
  )

print(p3)

## ----reset-theme, fig.alt="Scatterplot of engine performance analysis with default ggplot2 theme"----
# Reset to original theme
brand_reset_ggplot()

# Create a plot with default theme to show the difference
p4 <- ggplot(mtcars, aes(x = mpg, y = wt)) +
  geom_point(aes(color = factor(cyl)), size = 3) +
  labs(
    title = "Same Plot with Default Theme",
    x = "Miles per Gallon",
    y = "Weight (1000 lbs)",
    color = "Cylinders"
  )

print(p4)

## ----troubleshooting, eval=FALSE----------------------------------------------
# # Disable custom fonts if having issues
# brand_set_ggplot(use_fonts = FALSE)
# 
# # Check brand configuration
# doc <- yaml::read_yaml("_brand.yml")
# str(doc$color)
# str(doc$typography)

## ----cleanup, include = FALSE-------------------------------------------------
file.remove("_brand.yml")

