## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.dim = c(6, 4)
)

## ----setup--------------------------------------------------------------------
library(EZbakR)
library(dplyr)
set.seed(42)

## ----eval = FALSE-------------------------------------------------------------
# # Check out examples:
# example_cB
# example_metadf

## -----------------------------------------------------------------------------
ezbdo <- EZbakRData(example_cB, example_metadf)

## -----------------------------------------------------------------------------
# Simulate 500 "genes" worth of data
  # Simulates 2 replicates of 2 different experimental conditions
simdata <- EZSimulate(nfeatures = 500, nreps = 2)

ezbdo <- EZbakRData(simdata$cB, simdata$metadf)

## -----------------------------------------------------------------------------
ezbdo <- EstimateFractions(ezbdo, features = "feature")

### NOTE:
### For real data from fastq2EZbakR, you will likely
### want to set `features = "XF"`. See EstimateFractions()
### vignette for details

## ----fig.align='center'-------------------------------------------------------
# Simulated ground truth
truth <- simdata$PerRepTruth

# Get fraction estimates. 
  # Can also access with ezbdo$fractions[[1]] or ezbdo$fractions$feature
est <- EZget(ezbdo, type = 'fractions')

# Combine
compare <- dplyr::inner_join(est, truth, by = c('sample', 'feature'))

# Assess accuracy
plot(compare$true_fraction_highTC, compare$fraction_highTC)
abline(0,1)

## -----------------------------------------------------------------------------
ezbdo <- EstimateKinetics(ezbdo)

## ----fig.align='center'-------------------------------------------------------
# Simulated ground truth
truth <- simdata$PerRepTruth

# Get fraction estimates. 
  # Can also access with ezbdo$fractions[[1]] or ezbdo$fractions$feature
est <- EZget(ezbdo, type = 'kinetics')

# Combine
compare <- dplyr::inner_join(est, truth, by = c('sample', 'feature'))

# Assess accuracy
plot(log(compare$true_kdeg), compare$log_kdeg)
abline(0,1)

## -----------------------------------------------------------------------------
ezbdo <- AverageAndRegularize(ezbdo,
                              parameter = "log_kdeg")

## -----------------------------------------------------------------------------
ezbdo <- AverageAndRegularize(ezbdo,
                              parameter = "log_ksyn")

## ----fig.align='center'-------------------------------------------------------
truth <- simdata$AvgTruth

est <- EZget(ezbdo, type = 'averages', parameter = 'log_kdeg')

compare <- dplyr::inner_join(truth, est, by = 'feature')

plot(compare$true_logkdeg_treatmenttreatment1,
     compare$mean_treatmenttreatment1)
abline(0,1)


## -----------------------------------------------------------------------------
ezbdo <- CompareParameters(ezbdo,
                           condition = 'treatment',
                           reference = 'treatment1',
                           experimental = 'treatment2',
                           parameter = "log_kdeg")

## ----fig.align='center'-------------------------------------------------------
EZVolcanoPlot(ezbdo, parameter = "log_kdeg",
              condition = 'treatment',
              reference = 'treatment1',
              experimental = 'treatment2')

## ----fig.align='center'-------------------------------------------------------
EZMAPlot(ezbdo, parameter = "log_kdeg",
         condition = 'treatment',
         reference = 'treatment1',
         experimental = 'treatment2')

