## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----test_data----------------------------------------------------------------
library(ambiR)

head(test_data)

## ----transpose_wide, include=FALSE--------------------------------------------
library(dplyr)
library(tidyr)

wide_data_species <- test_data %>%
  pivot_wider(names_from = "species", 
              values_from = "count",
              values_fill = 0)

## ----data_wide_spec-----------------------------------------------------------
head(wide_data_species)

## ----pivot_species------------------------------------------------------------
# columns 1 and 2 contain station and replicate information
# so, select all columns from 3 to be pivoted 

long_data <- wide_data_species %>%
  pivot_longer(cols = 3:ncol(wide_data_species), 
               names_to = "species",
               values_to = "count")

head(long_data)

## ----transpose_wide_stns, include=FALSE---------------------------------------
df <- test_data %>%
  pivot_wider(names_from = c("station","replicate"), 
              values_from = "count", values_fill = 0)

stns <- names(df)[2:ncol(df)]

stns <- stns %>% 
  sapply(strsplit,"_") 
reps <- stns  %>%  
  sapply(function(x){x[2]})
stns <- stns  %>%  
  sapply(function(x){x[1]})
n <- 1+length(stns)
df2 <- matrix(nrow=2, ncol=n) %>% 
  as.data.frame()

df2[1,2:n] <- stns
df2[2,2:n] <- reps
names(df) <- names(df2)
df <- df %>%
  mutate(across(all_of(names(df)), as.character))

df <- df2 %>%
  bind_rows(df)

names(df) <- rep("", ncol(df))

wide_data_stns <- df

## ----data_wide_stns-----------------------------------------------------------
head(wide_data_stns)

## ----combine_stn_rep----------------------------------------------------------
sep_character <- "_"

# get the station IDs from row 1, excluding column 1 (this contains species names)
stations <- wide_data_stns[1, 2:ncol(wide_data_stns)]

# get the replicate IDs from row 2
replicates <- wide_data_stns[2, 2:ncol(wide_data_stns)]

# combine the station and replicate for each column using an underscore
station_replicate <- paste(stations, replicates, sep=sep_character)

# now we have extracted the station/replicate information, we can drop the 
# first two rows of the data frame
wide_data_stns <- wide_data_stns[3:nrow(wide_data_stns),]

# apply the station_replicates as column names
names(wide_data_stns) <- c("species", station_replicate)

head(wide_data_stns)

## ----pivot_stations-----------------------------------------------------------
# column 1 contains species names
# so, select all columns from 2 to be pivoted 

long_data <- wide_data_stns %>%
  pivot_longer(cols = 2:ncol(wide_data_stns), 
               names_to = "stn_rep",
               values_to = "count")

head(long_data)

## ----split_stations-----------------------------------------------------------
long_data <- long_data %>%
  separate_wider_delim(cols="stn_rep", 
                       delim = sep_character,
                       names=c("station", "replicate"))

head(long_data)

## ----test_data2---------------------------------------------------------------
head(test_data)

## ----run_ambi-----------------------------------------------------------------
res <- AMBI(test_data, by="station", var_rep="replicate", 
            var_species="species", var_count="count")

## ----show_ambi----------------------------------------------------------------
res$AMBI

## ----show_ambi_rep------------------------------------------------------------
res$AMBI_rep

## ----show_matched-------------------------------------------------------------
head(res$matched)

## ----limits_mambi-------------------------------------------------------------
limits_AMBI <- c("bad" = 6, "high" = 0)

limits_H  <- c("bad" = 0, "high" = NA)

limits_S  <- c("bad" = 0, "high" = NA)

## ----bounds_mambi-------------------------------------------------------------
bounds <-c("PB" = 0.2, "MP" = 0.39, "GM" = 0.53, "HG" = 0.77)

## ----calc_mambi---------------------------------------------------------------
res_mambi <- MAMBI(res$AMBI, var_H = "H", var_S = "S", var_AMBI = "AMBI")

## ----mambi_results------------------------------------------------------------
res_mambi %>%
  select(station, AMBI, H, S, x, y, z, MAMBI, Status, EQR)

